//----------------------------------------------------clipboard panel control----------------------------------------------------------------------//
// Function to update clipboard and preview widths based on canvas width
function updatePanelWidths() {
    const canvas = document.getElementById('canvas');
    const clipboardPanel = document.getElementById('clipboardPanel');
    const previewPanel = document.getElementById('previewPanel');
    const logContainer = document.getElementById('clipboard');

    if (canvas && clipboardPanel && previewPanel && logContainer) {
        // Get canvas width
        const canvasWidth = canvas.width;

        // Set log container width to match canvas width
        logContainer.style.width = canvasWidth + 'px';

        // Set clipboard panel to 70% of canvas width
        const clipboardWidth = canvasWidth * 0.7 - 40; // 40px for padding and gap
        clipboardPanel.style.width = clipboardWidth + 'px';

        // Set preview panel to 30% of canvas width
        const previewWidth = canvasWidth * 0.3 - 40; // 40px for padding and gap
        previewPanel.style.width = previewWidth + 'px';

        // Also update the max-width for responsiveness
        logContainer.style.maxWidth = '100%';
    }
}
// Initial update
updatePanelWidths();
//----------------------------------------------------clipboard panel control----------------------------------------------------------------------//
// 精确文字大小调整函数 - 针对特定元素
function adjustSpecificTextSizes(xScreen) {
    scaleFactor = 1.0;
    if (xScreen >= 2000)
        scaleFactor = 1.5;

    // 定义需要调整的具体元素ID和类名
    const specificElements = [
        // 按钮元素
        '#helpBtn', '#clearLogBtn', '#saveLogBtn',
        '#applySettingsBtn', '#closeSettingsBtn', '#fullscreenBtn',
        '#refreshBtn', '#screenshotBtn',

        // 控制按钮
        '.control-btn', '.btn-secondary', '.canvas-btn',

        // 面板标题和文本
        '.panel-title', '.panel-header', '.setting-label',
        '.connection-status', '.status-text', '#settingsText', '#helpText',
        '.website-link-text',

        // 特定文本元素
        '#info_txt', '.logo-text', '.point-hand',

        // 表单元素
        'textarea',

        '#footer'
    ];

    specificElements.forEach(selector => {
        const elements = document.querySelectorAll(selector);

        elements.forEach(element => {           
            const computedStyle = window.getComputedStyle(element);
            let currentSize = parseFloat(computedStyle.fontSize);

            // 如果无法获取字体大小，使用默认值
            if (isNaN(currentSize)) {
                currentSize = 20;
            }

            // 应用缩放因子
            const newSize = currentSize * scaleFactor;

            // 设置新的字体大小
            element.style.fontSize = `${newSize}px`;

            // 对于按钮，可能需要调整内边距
            if (selector.includes('btn') || selector.includes('Btn')) {
                const padding = parseFloat(computedStyle.paddingTop) || 0;
                if (padding > 0) {
                    const newPadding = padding * scaleFactor;
                    element.style.padding = `${newPadding}px ${newPadding * 1.5}px`;
                }
            }
        });
    });

    console.log(`特定元素文字大小已调整，缩放因子: ${scaleFactor}`);
}
//=============================================save log===============================================================================

function showInfoMessage(message, type = 'info') {
    const infoTxt = document.getElementById('info_txt');
    if (infoTxt) {
        infoTxt.textContent = message;

        if (type === 'warning') {
            infoTxt.style.color = '#f59e0b';
            infoTxt.style.background = 'rgba(245, 158, 11, 0.1)';
        } else if (type === 'error') {
            infoTxt.style.color = '#ef4444';
            infoTxt.style.background = 'rgba(239, 68, 68, 0.1)';
        } else {
            infoTxt.style.color = '#3b82f6';
            infoTxt.style.background = 'rgba(59, 130, 246, 0.1)';
        }

        infoTxt.style.display = 'block';

        setTimeout(() => {
            infoTxt.style.display = 'none';
        }, 3000);
    } else {
        alert(message);
    }
}


function secureDownload(content, filename, contentType) {
    // 方法1: 使用数据URL（适用于较小文件）
    if (content.length < 32768) { // 约32KB限制
        const dataUrl = `data:${contentType};charset=utf-8,${encodeURIComponent(content)}`;
        const a = document.createElement('a');
        a.href = dataUrl;
        a.download = filename;
        a.style.display = 'none';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);

        showInfoMessage('日志已成功保存为 ' + filename);
        return;
    }

    // 方法2: 对于大文件，使用Blob但添加安全警告
    if (window.location.protocol === 'http:') {
        const shouldContinue = confirm(
            '您的连接不安全。在HTTP上下文中保存大文件可能会导致浏览器警告。\n\n' +
            '要继续保存文件吗？'
        );

        if (!shouldContinue) return;
    }

    try {
        const blob = new Blob([content], { type: contentType });
        const url = URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = filename;
        a.style.display = 'none';
        document.body.appendChild(a);
        a.click();

        // 清理
        setTimeout(() => {
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
            showInfoMessage('日志已成功保存为 ' + filename);
        }, 100);
    } catch (e) {
        console.error('保存文件时出错:', e);
        alert('保存文件时出错: ' + e.message);
    }
}

function setupSecureLogSaveFunction() {
    const saveLogBtn = document.getElementById('saveLogBtn');
    const textarea = document.getElementById('textarea');

    if (!saveLogBtn || !textarea) {
        console.warn('No saveLogBtn');
        return;
    }

    saveLogBtn.addEventListener('click', function () {
        const logContent = textarea.value;
        if (!logContent.trim()) {
            alert('Empty log');
            return;
        }
              
        const timestamp = new Date().toISOString()
            .replace(/[:.]/g, '-')
            .replace('T', '_')
            .substring(0, 19);

        const filename = `SIMCommander_Log_${timestamp}.txt`;
        secureDownload(logContent, filename, 'text/plain');
    });
}

// 在DOM加载完成后初始化
document.addEventListener('DOMContentLoaded', function () {
    setupSecureLogSaveFunction();
});
//========================================save log==================================================================================


//========================================save canvas==================================================================================//
function setupCanvasScreenshot() {
    const screenshotBtn = document.getElementById('screenshotBtn');
    const canvas = document.getElementById('canvas');

    if (!screenshotBtn || !canvas) {
        console.warn('截图按钮或Canvas元素未找到');
        return;
    }

    screenshotBtn.addEventListener('click', function () {
        try {
            // 检查Canvas是否为空或太小
            if (canvas.width === 0 || canvas.height === 0) {
                alert('Canvas内容为空，无法截图');
                return;
            }

            // 创建时间戳用于文件名
            const timestamp = new Date().toISOString()
                .replace(/[:.]/g, '-')
                .replace('T', '_')
                .substring(0, 19);

            // 创建文件名
            const filename = `SIMCommander_Screenshot_${timestamp}.png`;

            // 将Canvas转换为数据URL
            const dataURL = canvas.toDataURL('image/png');

            // 创建下载链接
            const a = document.createElement('a');
            a.href = dataURL;
            a.download = filename;
            a.style.display = 'none';

            // 添加到文档并模拟点击
            document.body.appendChild(a);
            a.click();

            // 清理
            setTimeout(() => {
                document.body.removeChild(a);

                // 显示成功消息
                const infoTxt = document.getElementById('info_txt');
                if (infoTxt) {
                    infoTxt.textContent = 'Image was saved to ' + filename;
                    infoTxt.style.display = 'block';
                    setTimeout(() => {
                        infoTxt.style.display = 'none';
                    }, 3000);
                }
            }, 100);

        } catch (error) {
            console.error('截图失败:', error);
            alert('截图失败: ' + error.message);
        }
    });
}

// 在DOM加载完成后初始化
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', setupCanvasScreenshot);
} else {
    setupCanvasScreenshot();
}
//========================================save canvas==================================================================================//


//========================================clipboard to clipboard and clear==================================================================================//
function setupClipboardFunctions() {
    // 获取剪贴板相关元素
    const copyBtn = document.querySelector('[title="Copy to Clipboard"]');
    const clearBtn = document.querySelector('[title="Clear Clipboard"]');
    const textarea = document.getElementById('textarea');

    if (!copyBtn || !clearBtn || !textarea) {
        console.warn('剪贴板按钮或文本区域未找到');
        return;
    }

    // 复制到剪贴板功能
    copyBtn.addEventListener('click', async function () {
        try {
            // 使用现代Clipboard API
            await navigator.clipboard.writeText(textarea.value);

            // 显示成功消息
            showInfoMessage('内容已复制到剪贴板');

            // 添加视觉反馈
            copyBtn.innerHTML = '<i class="fas fa-check"></i>';
            setTimeout(() => {
                copyBtn.innerHTML = '<i class="fas fa-copy"></i>';
            }, 2000);

        } catch (err) {
            console.error('复制到剪贴板失败:', err);

            // 回退到传统方法
            fallbackCopyText(textarea);
        }
    });

    // 清除剪贴板内容功能
    clearBtn.addEventListener('click', function () {
        // 保存当前滚动位置
        const scrollPosition = textarea.scrollTop;

        // 清空文本区域
        textarea.value = '';

        // 恢复滚动位置
        textarea.scrollTop = scrollPosition;

        // 添加视觉反馈
        clearBtn.innerHTML = '<i class="fas fa-check"></i>';
        setTimeout(() => {
            clearBtn.innerHTML = '<i class="fas fa-trash-alt"></i>';
        }, 2000);
    });
}

// 传统的复制到剪贴板方法（作为Clipboard API的备用方案）
function fallbackCopyText(textarea) {
    // 选择文本区域的内容
    textarea.select();
    textarea.setSelectionRange(0, 99999); // 对于移动设备

    try {
        // 执行复制命令
        const successful = document.execCommand('copy');

        if (successful) {
            showInfoMessage('内容已复制到剪贴板');
        } else {
            throw new Error('复制命令失败');
        }
    } catch (err) {
        console.error('传统复制方法失败:', err);

        // 最终备用方案：提示用户手动复制
        textarea.select();
        showInfoMessage('请按Ctrl+C复制内容', 'warning');
    }
}

document.addEventListener('DOMContentLoaded', function () {
    setupClipboardFunctions();
});

//========================================clipboard to clipboard and clear==================================================================================//
