// sha256.js - 纯JavaScript SHA-256实现

// SHA-256常量定义
const k = new Uint32Array([
    0x428a2f98, 0x71374491, 0xb5c0fbcf, 0xe9b5dba5, 0x3956c25b, 0x59f111f1, 0x923f82a4, 0xab1c5ed5,
    0xd807aa98, 0x12835b01, 0x243185be, 0x550c7dc3, 0x72be5d74, 0x80deb1fe, 0x9bdc06a7, 0xc19bf174,
    0xe49b69c1, 0xefbe4786, 0x0fc19dc6, 0x240ca1cc, 0x2de92c6f, 0x4a7484aa, 0x5cb0a9dc, 0x76f988da,
    0x983e5152, 0xa831c66d, 0xb00327c8, 0xbf597fc7, 0xc6e00bf3, 0xd5a79147, 0x06ca6351, 0x14292967,
    0x27b70a85, 0x2e1b2138, 0x4d2c6dfc, 0x53380d13, 0x650a7354, 0x766a0abb, 0x81c2c92e, 0x92722c85,
    0xa2bfe8a1, 0xa81a664b, 0xc24b8b70, 0xc76c51a3, 0xd192e819, 0xd6990624, 0xf40e3585, 0x106aa070,
    0x19a4c116, 0x1e376c08, 0x2748774c, 0x34b0bcb5, 0x391c0cb3, 0x4ed8aa4a, 0x5b9cca4f, 0x682e6ff3,
    0x748f82ee, 0x78a5636f, 0x84c87814, 0x8cc70208, 0x90befffa, 0xa4506ceb, 0xbef9a3f7, 0xc67178f2
]);

// 初始哈希值
const h0 = new Uint32Array([
    0x6a09e667, 0xbb67ae85, 0x3c6ef372, 0xa54ff53a,
    0x510e527f, 0x9b05688c, 0x1f83d9ab, 0x5be0cd19
]);

// 右旋转函数
function ROTRIGHT(word, bits) {
    return ((word >>> bits) | (word << (32 - bits))) >>> 0;
}

// SHA-256辅助函数
function CH(x, y, z) {
    return ((x & y) ^ (~x & z)) >>> 0;
}

function MAJ(x, y, z) {
    return ((x & y) ^ (x & z) ^ (y & z)) >>> 0;
}

function EP0(x) {
    return (ROTRIGHT(x, 2) ^ ROTRIGHT(x, 13) ^ ROTRIGHT(x, 22)) >>> 0;
}

function EP1(x) {
    return (ROTRIGHT(x, 6) ^ ROTRIGHT(x, 11) ^ ROTRIGHT(x, 25)) >>> 0;
}

function SIG0(x) {
    return (ROTRIGHT(x, 7) ^ ROTRIGHT(x, 18) ^ (x >>> 3)) >>> 0;
}

function SIG1(x) {
    return (ROTRIGHT(x, 17) ^ ROTRIGHT(x, 19) ^ (x >>> 10)) >>> 0;
}

// SHA-256上下文类
class SHA256_CTX {
    constructor() {
        this.data = new Uint8Array(64);
        this.datalen = 0;
        this.bitlen = 0n; // 使用BigInt处理64位整数
        this.state = new Uint32Array(h0); // 复制初始值
    }
}

// 初始化SHA-256上下文
function sha256_init(ctx) {
    ctx.datalen = 0;
    ctx.bitlen = 0n;
    ctx.state.set(h0); // 重置为初始哈希值
}

// 处理单个512位数据块
function sha256_transform(ctx, data) {
    let a, b, c, d, e, f, g, h, i, j, t1, t2;
    const m = new Uint32Array(64);

    // 将数据从大端序转换为小端序并填充到消息数组中
    for (i = 0, j = 0; i < 16; ++i, j += 4) {
        m[i] = (data[j] << 24) | (data[j + 1] << 16) | (data[j + 2] << 8) | (data[j + 3]);
        m[i] >>>= 0; // 确保是32位无符号整数
    }

    // 扩展消息数组
    for (; i < 64; ++i) {
        m[i] = (SIG1(m[i - 2]) + m[i - 7] + SIG0(m[i - 15]) + m[i - 16]) >>> 0;
    }

    // 初始化工作变量
    a = ctx.state[0];
    b = ctx.state[1];
    c = ctx.state[2];
    d = ctx.state[3];
    e = ctx.state[4];
    f = ctx.state[5];
    g = ctx.state[6];
    h = ctx.state[7];

    // 主循环
    for (i = 0; i < 64; ++i) {
        t1 = (h + EP1(e) + CH(e, f, g) + k[i] + m[i]) >>> 0;
        t2 = (EP0(a) + MAJ(a, b, c)) >>> 0;
        h = g;
        g = f;
        f = e;
        e = (d + t1) >>> 0;
        d = c;
        c = b;
        b = a;
        a = (t1 + t2) >>> 0;
    }

    // 更新哈希值
    ctx.state[0] = (ctx.state[0] + a) >>> 0;
    ctx.state[1] = (ctx.state[1] + b) >>> 0;
    ctx.state[2] = (ctx.state[2] + c) >>> 0;
    ctx.state[3] = (ctx.state[3] + d) >>> 0;
    ctx.state[4] = (ctx.state[4] + e) >>> 0;
    ctx.state[5] = (ctx.state[5] + f) >>> 0;
    ctx.state[6] = (ctx.state[6] + g) >>> 0;
    ctx.state[7] = (ctx.state[7] + h) >>> 0;
}

// 更新SHA-256上下文
function sha256_update(ctx, inputData) {
    const len = inputData.length;
    
    for (let i = 0; i < len; ++i) {
        ctx.data[ctx.datalen] = inputData[i];
        ctx.datalen++;

        if (ctx.datalen === 64) {
            sha256_transform(ctx, ctx.data);
            ctx.bitlen += 512n;
            ctx.datalen = 0;
        }
    }
}

// 完成SHA-256计算 - 特别优化空文本处理
function sha256_final(ctx) {
    const hash = new Uint8Array(32);
    let i = ctx.datalen;

    // 处理空文本的特殊情况
    if (ctx.datalen === 0 && ctx.bitlen === 0n) {
        // 空文本需要特殊处理：只有填充和长度
        ctx.data[i++] = 0x80; // 1后面跟0的填充开始
        
        // 填充到56字节（为长度留8字节空间）
        while (i < 56) {
            ctx.data[i++] = 0x00;
        }
    } 
    // 正常情况下的填充
    else if (ctx.datalen < 56) {
        ctx.data[i++] = 0x80;
        while (i < 56) {
            ctx.data[i++] = 0x00;
        }
    } else {
        ctx.data[i++] = 0x80;
        while (i < 64) {
            ctx.data[i++] = 0x00;
        }
        sha256_transform(ctx, ctx.data);
        // 清空数据块
        for (let j = 0; j < 56; j++) {
            ctx.data[j] = 0x00;
        }
    }

    // 添加位长度（64位大端序）
    ctx.bitlen += BigInt(ctx.datalen * 8);
    
    // 将64位长度写入最后8个字节（大端序）
    for (let j = 7; j >= 0; j--) {
        ctx.data[56 + j] = Number((ctx.bitlen >> BigInt(8 * (7 - j))) & 0xFFn);
    }

    sha256_transform(ctx, ctx.data);

    // 将哈希值从大端序转换为小端序输出
    for (i = 0; i < 4; ++i) {
        hash[i] = (ctx.state[0] >> (24 - i * 8)) & 0xff;
        hash[i + 4] = (ctx.state[1] >> (24 - i * 8)) & 0xff;
        hash[i + 8] = (ctx.state[2] >> (24 - i * 8)) & 0xff;
        hash[i + 12] = (ctx.state[3] >> (24 - i * 8)) & 0xff;
        hash[i + 16] = (ctx.state[4] >> (24 - i * 8)) & 0xff;
        hash[i + 20] = (ctx.state[5] >> (24 - i * 8)) & 0xff;
        hash[i + 24] = (ctx.state[6] >> (24 - i * 8)) & 0xff;
        hash[i + 28] = (ctx.state[7] >> (24 - i * 8)) & 0xff;
    }

    return hash;
}

// 计算数据的SHA-256哈希值（返回Uint8Array）
function sha256_bytes(input) {
    const ctx = new SHA256_CTX();
    sha256_init(ctx);
    
    // 如果输入是字符串，转换为字节数组
    let data;
    if (typeof input === 'string') {
        const encoder = new TextEncoder();
        data = encoder.encode(input);
    } else {
        data = new Uint8Array(input);
    }
    
    // 即使是空数组也要调用update，以确保bitlen正确设置
    sha256_update(ctx, data);
    return sha256_final(ctx);
}

// 计算SHA-256并返回十六进制字符串
function sha256_hex(input) {
    const hashBytes = sha256_bytes(input);
    return Array.from(hashBytes)
        .map(b => b.toString(16).padStart(2, '0'))
        .join('');
}

// 导出函数（根据环境选择导出方式）
if (typeof module !== 'undefined' && module.exports) {
    // Node.js环境
    module.exports = {
        sha256_bytes,
        sha256_hex,
        SHA256_CTX,
        sha256_init,
        sha256_update,
        sha256_final,
        sha256_transform
    };
} else {
    // 浏览器环境 - 挂载到window对象
    window.sha256_bytes = sha256_bytes;
    window.sha256_hex = sha256_hex;
    window.SHA256_CTX = SHA256_CTX;
    window.sha256_init = sha256_init;
    window.sha256_update = sha256_update;
    window.sha256_final = sha256_final;
    window.sha256_transform = sha256_transform;
}